/**
 * ContextBridge - Shared Utilities
 */

const CBUtils = {
  hash(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      const char = str.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash;
    }
    return hash.toString(36);
  },

  generateId() {
    return 'cb_' + Date.now().toString(36) + '_' + Math.random().toString(36).substring(2, 8);
  },

  debounce(fn, delay) {
    let timer = null;
    return function (...args) {
      clearTimeout(timer);
      timer = setTimeout(() => fn.apply(this, args), delay);
    };
  },

  timeAgo(dateStr) {
    const now = Date.now();
    const then = new Date(dateStr).getTime();
    const diff = now - then;
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);

    if (minutes < 1) return 'just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 30) return `${days}d ago`;
    return new Date(dateStr).toLocaleDateString();
  },

  truncate(str, maxLen = 50) {
    if (!str || str.length <= maxLen) return str;
    return str.substring(0, maxLen) + '...';
  },

  escapeHtml(str) {
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
  },

  /**
   * Format context entries into the injection block
   * @param {Array} entries - context entries
   * @param {number} maxLength - max total chars
   * @returns {string} formatted context block
   */
  formatContextBlock(entries, maxLength = 2000) {
    const { CB_CATEGORY_ORDER, CB_CATEGORY_NAMES } = window.ContextBridge;

    if (!entries || entries.length === 0) return '';

    // Group entries by category
    const grouped = {};
    for (const entry of entries) {
      if (!grouped[entry.category]) grouped[entry.category] = [];
      grouped[entry.category].push(entry);
    }

    // Build block in priority order
    let block = '[CONTEXT START]\n';
    let currentLength = block.length + '\n[CONTEXT END]'.length;

    for (const category of CB_CATEGORY_ORDER) {
      if (!grouped[category] || grouped[category].length === 0) continue;

      const header = `\n## ${CB_CATEGORY_NAMES[category]}\n`;
      if (currentLength + header.length > maxLength) break;

      block += header;
      currentLength += header.length;

      for (const entry of grouped[category]) {
        const line = `- ${entry.text}\n`;
        if (currentLength + line.length > maxLength) break;
        block += line;
        currentLength += line.length;
      }
    }

    block += '[CONTEXT END]';
    return block;
  }
};

if (typeof window !== 'undefined') {
  window.ContextBridge = window.ContextBridge || {};
  window.ContextBridge.Utils = CBUtils;
}
