/**
 * ContextBridge - Popup Controller
 * Manages context entries, suggestions, settings, and export/import.
 */

const CATEGORIES = ['rules', 'preferences', 'projects', 'tasks', 'decisions', 'people'];
const CATEGORY_LABELS = {
  rules: 'Rule', preferences: 'Preference', projects: 'Project',
  tasks: 'Task', decisions: 'Decision', people: 'Person'
};

const PLATFORM_NAMES = {
  chatgpt: 'ChatGPT', claude: 'Claude', gemini: 'Gemini',
  copilot: 'Copilot', character_ai: 'Character.AI', deepseek: 'DeepSeek',
  grok: 'Grok', pi: 'Pi', poe: 'Poe', perplexity: 'Perplexity',
  huggingchat: 'HuggingChat', lechat: 'Le Chat'
};

let currentEntries = [];
let currentSettings = {};
let modalState = null; // { mode: 'add'|'edit', category, entryId }

// --- Init ---

document.addEventListener('DOMContentLoaded', async () => {
  await loadAll();
  setupEventListeners();
});

async function loadAll() {
  await Promise.all([loadEntries(), loadSuggestions(), loadStats(), loadSettings()]);
}

// --- Data Loading ---

async function loadEntries() {
  const response = await sendMessage('GET_ALL_ENTRIES');
  if (!response.success) return;
  currentEntries = response.entries || [];
  renderEntries();
  updateCharCount();
}

async function loadSuggestions() {
  const response = await sendMessage('GET_SUGGESTIONS');
  if (!response.success) return;
  renderSuggestions(response.suggestions || []);
}

async function loadStats() {
  const response = await sendMessage('GET_STATS');
  if (!response.success) return;

  document.getElementById('stat-entries').textContent = `${response.entryCount} items`;
  document.getElementById('stat-injections').textContent = `${response.stats.totalInjections} injections`;
}

async function loadSettings() {
  const response = await sendMessage('GET_SETTINGS');
  if (!response.success) return;
  currentSettings = response.settings;

  document.getElementById('setting-auto-inject').checked = currentSettings.autoInject;
  document.getElementById('setting-auto-suggest').checked = currentSettings.autoSuggest;
  document.getElementById('setting-max-length').value = currentSettings.maxContextLength;
  document.getElementById('max-length-display').textContent = currentSettings.maxContextLength;

  renderPlatformToggles();
  updateCharCount();
}

// --- Rendering ---

function renderEntries() {
  for (const category of CATEGORIES) {
    const list = document.querySelector(`[data-list="${category}"]`);
    const countEl = document.querySelector(`[data-cat="${category}"]`);
    const catEntries = currentEntries.filter(e => e.category === category);

    countEl.textContent = catEntries.length;
    list.innerHTML = '';

    for (const entry of catEntries) {
      const row = document.createElement('div');
      row.className = 'entry-row';
      row.innerHTML = `
        <span class="entry-text">${escapeHtml(entry.text)}</span>
        <div class="entry-actions">
          <button class="entry-btn edit" data-id="${entry.id}" title="Edit">&#9998;</button>
          <button class="entry-btn delete" data-id="${entry.id}" title="Delete">&#10005;</button>
        </div>
      `;
      list.appendChild(row);
    }
  }
}

function renderSuggestions(suggestions) {
  const section = document.getElementById('suggestions-section');
  const body = document.getElementById('suggestions-body');
  const countEl = document.getElementById('suggestion-count');

  if (suggestions.length === 0) {
    section.style.display = 'none';
    return;
  }

  section.style.display = '';
  countEl.textContent = suggestions.length;
  body.innerHTML = '';

  for (const sug of suggestions) {
    const row = document.createElement('div');
    row.className = 'suggestion-row';
    row.innerHTML = `
      <div style="flex:1">
        <div class="suggestion-text">${escapeHtml(sug.text)}</div>
        <div class="suggestion-category">${sug.category} &middot; from ${PLATFORM_NAMES[sug.sourcePlatform] || sug.sourcePlatform}</div>
      </div>
      <div class="suggestion-actions">
        <button class="sug-btn approve" data-id="${sug.id}" title="Approve">+</button>
        <button class="sug-btn dismiss" data-id="${sug.id}" title="Dismiss">&times;</button>
      </div>
    `;
    body.appendChild(row);
  }
}

function renderPlatformToggles() {
  const container = document.getElementById('platform-toggles');
  container.innerHTML = '';

  for (const [key, name] of Object.entries(PLATFORM_NAMES)) {
    const enabled = currentSettings.platformsEnabled?.[key] !== false;
    const row = document.createElement('label');
    row.className = 'platform-toggle';
    row.innerHTML = `
      <span>${name}</span>
      <input type="checkbox" data-platform="${key}" ${enabled ? 'checked' : ''}>
    `;
    container.appendChild(row);
  }
}

async function updateCharCount() {
  const response = await sendMessage('FORMAT_CONTEXT');
  if (!response.success) return;
  const len = (response.block || '').length;
  const max = currentSettings.maxContextLength || 2000;
  document.getElementById('stat-chars').textContent = `${len}/${max} ch`;
}

// --- Event Listeners ---

function setupEventListeners() {
  // Section accordion toggles
  document.querySelectorAll('.section-header').forEach(header => {
    header.addEventListener('click', () => {
      const sectionName = header.dataset.section;
      let body;

      if (sectionName === 'suggestions') {
        body = document.getElementById('suggestions-body');
      } else {
        body = document.querySelector(`[data-body="${sectionName}"]`);
      }

      if (body) {
        body.classList.toggle('collapsed');
        const chevron = header.querySelector('.chevron');
        if (chevron) {
          chevron.innerHTML = body.classList.contains('collapsed') ? '&#9654;' : '&#9660;';
        }
      }
    });
  });

  // Add buttons
  document.querySelectorAll('.add-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      const category = btn.dataset.add;
      openModal('add', category);
    });
  });

  // Entry edit/delete (delegated)
  document.addEventListener('click', async (e) => {
    const editBtn = e.target.closest('.entry-btn.edit');
    if (editBtn) {
      const id = editBtn.dataset.id;
      const entry = currentEntries.find(e => e.id === id);
      if (entry) openModal('edit', entry.category, entry);
      return;
    }

    const deleteBtn = e.target.closest('.entry-btn.delete');
    if (deleteBtn) {
      const id = deleteBtn.dataset.id;
      await sendMessage('DELETE_ENTRY', { id });
      await loadAll();
      return;
    }

    // Suggestion approve/dismiss
    const approveBtn = e.target.closest('.sug-btn.approve');
    if (approveBtn) {
      await sendMessage('APPROVE_SUGGESTION', { id: approveBtn.dataset.id });
      await loadAll();
      return;
    }

    const dismissBtn = e.target.closest('.sug-btn.dismiss');
    if (dismissBtn) {
      await sendMessage('DISMISS_SUGGESTION', { id: dismissBtn.dataset.id });
      await loadAll();
      return;
    }
  });

  // Modal
  document.getElementById('modal-cancel').addEventListener('click', closeModal);
  document.getElementById('modal-save').addEventListener('click', saveModal);
  document.getElementById('modal-overlay').addEventListener('click', (e) => {
    if (e.target === e.currentTarget) closeModal();
  });

  // Settings toggle
  document.getElementById('btn-settings').addEventListener('click', () => {
    const panel = document.getElementById('settings-panel');
    const btn = document.getElementById('btn-settings');
    const visible = panel.style.display !== 'none';
    panel.style.display = visible ? 'none' : '';
    btn.classList.toggle('active', !visible);
  });

  // Settings changes
  document.getElementById('setting-auto-inject').addEventListener('change', saveSettingsFromUI);
  document.getElementById('setting-auto-suggest').addEventListener('change', saveSettingsFromUI);
  document.getElementById('setting-max-length').addEventListener('input', (e) => {
    document.getElementById('max-length-display').textContent = e.target.value;
  });
  document.getElementById('setting-max-length').addEventListener('change', saveSettingsFromUI);

  // Platform toggles (delegated)
  document.getElementById('platform-toggles').addEventListener('change', saveSettingsFromUI);

  // Export
  document.getElementById('btn-export').addEventListener('click', async () => {
    const response = await sendMessage('EXPORT_DATA');
    if (!response.success) return;

    const blob = new Blob([JSON.stringify(response.data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `contextbridge-export-${new Date().toISOString().slice(0,10)}.json`;
    a.click();
    URL.revokeObjectURL(url);
  });

  // Import
  document.getElementById('btn-import').addEventListener('click', () => {
    document.getElementById('import-file').click();
  });

  document.getElementById('import-file').addEventListener('change', async (e) => {
    const file = e.target.files[0];
    if (!file) return;

    try {
      const text = await file.text();
      const data = JSON.parse(text);
      await sendMessage('IMPORT_DATA', data);
      await loadAll();
    } catch (err) {
      console.error('Import failed:', err);
    }
    e.target.value = '';
  });

  // Clear all
  document.getElementById('btn-clear').addEventListener('click', async () => {
    if (confirm('Clear all context entries? This cannot be undone.')) {
      await sendMessage('CLEAR_ALL');
      await loadAll();
    }
  });
}

// --- Modal ---

function openModal(mode, category, entry = null) {
  modalState = { mode, category, entryId: entry?.id || null };

  const title = document.getElementById('modal-title');
  const input = document.getElementById('modal-input');

  if (mode === 'add') {
    title.textContent = `Add ${CATEGORY_LABELS[category]}`;
    input.value = '';
  } else {
    title.textContent = `Edit ${CATEGORY_LABELS[category]}`;
    input.value = entry?.text || '';
  }

  document.getElementById('modal-overlay').style.display = '';
  input.focus();
}

function closeModal() {
  document.getElementById('modal-overlay').style.display = 'none';
  modalState = null;
}

async function saveModal() {
  if (!modalState) return;

  const text = document.getElementById('modal-input').value.trim();
  if (!text) return;

  if (modalState.mode === 'add') {
    await sendMessage('ADD_ENTRY', { text, category: modalState.category });
  } else {
    await sendMessage('UPDATE_ENTRY', { id: modalState.entryId, updates: { text } });
  }

  closeModal();
  await loadAll();
}

// --- Settings ---

async function saveSettingsFromUI() {
  const platformsEnabled = {};
  document.querySelectorAll('#platform-toggles input[type="checkbox"]').forEach(cb => {
    platformsEnabled[cb.dataset.platform] = cb.checked;
  });

  const settings = {
    autoInject: document.getElementById('setting-auto-inject').checked,
    autoSuggest: document.getElementById('setting-auto-suggest').checked,
    maxContextLength: parseInt(document.getElementById('setting-max-length').value, 10),
    platformsEnabled
  };

  await sendMessage('SAVE_SETTINGS', settings);
  currentSettings = settings;
  updateCharCount();
}

// --- Helpers ---

function sendMessage(type, payload = {}) {
  return chrome.runtime.sendMessage({ type, payload });
}

function escapeHtml(str) {
  const div = document.createElement('div');
  div.textContent = str;
  return div.innerHTML;
}
