/**
 * ContextBridge - Poe Extractor
 * poe.com
 * Uses CSS Module hashed class names — requires substring matching.
 */

(function () {
  const { BaseExtractor, SelectorEngine } = window.ContextBridge;

  class PoeExtractor extends BaseExtractor {
    constructor() {
      super('poe');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'textarea[class*="GrowingTextArea"]',
          'textarea[class*="TextArea"]',
          'textarea'
        ],
        sendSelectors: [
          'button[class*="SendButton"]',
          'button[class*="sendButton"]',
          'button[aria-label="Send"]',
          'button[type="submit"]'
        ],
        inputType: 'textarea',
        messageContainer: [
          '[class*="ChatMessagesView_infiniteScroll"]',
          '[class*="ChatMessages"]',
          '[role="main"]',
          'main'
        ],
        userMessageSelectors: [
          '[class*="ChatMessage_messageRow"]',
          '[class*="messageRow"]'
        ],
        newConversationPattern: /^\/[^/]+\/?$/,
        periodicScan: false
      };
    }

    extractUserTexts(node) {
      const texts = [];

      const messageEls = this._getMessageElements(node);
      for (const el of messageEls) {
        if (this._isUserMessage(el)) {
          const text = this._extractText(el);
          if (text && text.length >= 10) texts.push(text);
        }
      }

      return texts;
    }

    _getMessageElements(node) {
      const cn = typeof node.className === 'string' ? node.className : '';
      if (cn.includes('messageRow') || cn.includes('MessageBubble')) return [node];
      const els = node.querySelectorAll?.('[class*="ChatMessage_messageRow"], [class*="messageRow"], [class*="MessageBubble"]') || [];
      return Array.from(els);
    }

    _isUserMessage(el) {
      const cn = typeof el.className === 'string' ? el.className : '';
      if (cn.includes('humanMessageBubble') || cn.includes('humanMessage') || cn.includes('rightSide')) return true;

      // Walk up
      let parent = el.parentElement;
      for (let i = 0; i < 5 && parent; i++) {
        const pcn = typeof parent.className === 'string' ? parent.className : '';
        if (pcn.includes('humanMessageBubble') || pcn.includes('humanMessage') || pcn.includes('rightSide')) return true;
        if (pcn.includes('botMessageBubble') || pcn.includes('botMessage') || pcn.includes('leftSide')) return false;
        parent = parent.parentElement;
      }
      return false;
    }

    _extractText(el) {
      const markdown = el.querySelector('[class*="Markdown_markdownContainer"]');
      if (markdown) return markdown.textContent.trim();
      const prose = el.querySelector('.prose');
      if (prose) return prose.textContent.trim();
      return (el.textContent || '').trim();
    }
  }

  const extractor = new PoeExtractor();
  extractor.init();
})();
