/**
 * ContextBridge - Le Chat (Mistral) Extractor
 * chat.mistral.ai
 * User messages in .select-none containers with span.whitespace-pre-wrap.
 */

(function () {
  const { BaseExtractor, SelectorEngine } = window.ContextBridge;

  class LeChatExtractor extends BaseExtractor {
    constructor() {
      super('lechat');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'textarea[placeholder]',
          'textarea'
        ],
        sendSelectors: [
          'button[data-testid="send-button"]',
          'button[aria-label="Send"]',
          'button[type="submit"]'
        ],
        inputType: 'textarea',
        messageContainer: ['body'],
        useBodyAsContainer: true,
        userMessageSelectors: [
          '.select-none'
        ],
        newConversationPattern: /^\/($|chat\/?$)/,
        periodicScan: true
      };
    }

    extractUserTexts(node) {
      const texts = [];

      if (this._isUserMessage(node)) {
        const text = this._extractText(node);
        if (text && text.length >= 10) texts.push(text);
        return texts;
      }

      const userEls = node.querySelectorAll?.('.select-none') || [];
      for (const el of userEls) {
        if (this._isUserMessage(el)) {
          const text = this._extractText(el);
          if (text && text.length >= 10) texts.push(text);
        }
      }

      return texts;
    }

    _isUserMessage(node) {
      const cn = typeof node.className === 'string' ? node.className : '';
      if (!cn.includes('select-none')) return false;
      // Must contain the user text span
      return !!node.querySelector?.('span.whitespace-pre-wrap');
    }

    _extractText(el) {
      const span = el.querySelector('span.whitespace-pre-wrap');
      if (span) return span.textContent.trim();
      return (el.textContent || '').trim();
    }
  }

  const extractor = new LeChatExtractor();
  extractor.init();
})();
