/**
 * ContextBridge - Grok Extractor
 * grok.com
 * User/AI differentiated by parent container class (.m-container.user vs .m-container.ai).
 */

(function () {
  const { BaseExtractor, SelectorEngine } = window.ContextBridge;

  class GrokExtractor extends BaseExtractor {
    constructor() {
      super('grok');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'textarea',
          'div[contenteditable="true"]'
        ],
        sendSelectors: [
          'button[aria-label="Send"]',
          'button[aria-label="Send message"]',
          'button[type="submit"]'
        ],
        inputType: 'auto',
        messageContainer: ['body'],
        useBodyAsContainer: true,
        userMessageSelectors: [
          '.message',
          '.m-container.user'
        ],
        newConversationPattern: /^\/($|chat\/?$)/,
        periodicScan: true
      };
    }

    extractUserTexts(node) {
      const texts = [];

      const messageEls = this._getMessageElements(node);
      for (const el of messageEls) {
        if (this._isUserMessage(el)) {
          const text = (el.textContent || '').trim();
          if (text.length >= 10) texts.push(text);
        }
      }

      return texts;
    }

    _getMessageElements(node) {
      const cn = typeof node.className === 'string' ? node.className : '';
      if (cn.includes('message') || cn.includes('m-container')) return [node];
      const els = node.querySelectorAll?.('.message, .m-container') || [];
      return Array.from(els);
    }

    _isUserMessage(el) {
      const cn = typeof el.className === 'string' ? el.className : '';
      if (cn.includes('m-container') && cn.includes('user')) return true;

      // Walk up to find parent m-container
      let parent = el.parentElement;
      for (let i = 0; i < 5 && parent; i++) {
        const pcn = typeof parent.className === 'string' ? parent.className : '';
        if (pcn.includes('m-container') && pcn.includes('user')) return true;
        if (pcn.includes('m-container') && pcn.includes('ai')) return false;
        parent = parent.parentElement;
      }
      return false;
    }
  }

  const extractor = new GrokExtractor();
  extractor.init();
})();
