/**
 * ContextBridge - Claude AI Extractor
 * claude.ai
 * Claude's SPA modifies existing DOM nodes — needs periodic scan.
 */

(function () {
  const { BaseExtractor } = window.ContextBridge;

  class ClaudeExtractor extends BaseExtractor {
    constructor() {
      super('claude');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'div.ProseMirror[contenteditable="true"]',
          'div[contenteditable="true"][data-placeholder]',
          'div[contenteditable="true"]'
        ],
        sendSelectors: [
          'button[aria-label="Send Message"]',
          'button[aria-label="Send message"]',
          'button[data-testid="send-button"]',
          'fieldset button[type="button"]'
        ],
        inputType: 'contenteditable',
        messageContainer: ['body'],
        useBodyAsContainer: true,
        userMessageSelectors: [
          'div[data-testid="user-message"]',
          'div[data-testid="chat-message-user"]'
        ],
        newConversationPattern: /^\/($|new|chat\/?$)/,
        periodicScan: true
      };
    }

    extractUserTexts(node) {
      const texts = [];

      // Check if node IS a user message
      if (this._isUserMessage(node)) {
        const text = (node.textContent || '').trim();
        if (text.length >= 10) texts.push(text);
        return texts;
      }

      // Check if node CONTAINS user messages
      const userEls = node.querySelectorAll?.('div[data-testid="user-message"]') || [];
      for (const el of userEls) {
        const text = (el.textContent || '').trim();
        if (text.length >= 10) texts.push(text);
      }

      return texts;
    }

    _isUserMessage(node) {
      if (!node?.getAttribute) return false;
      if (node.getAttribute('data-testid') === 'user-message') return true;
      if (node.getAttribute('data-testid') === 'chat-message-user') return true;
      const cn = typeof node.className === 'string' ? node.className : '';
      return cn.includes('font-user-message');
    }
  }

  const extractor = new ClaudeExtractor();
  extractor.init();
})();
