/**
 * ContextBridge - Character.AI Extractor
 * character.ai / beta.character.ai
 * User messages detected via flex-row-reverse alignment class.
 */

(function () {
  const { BaseExtractor, SelectorEngine } = window.ContextBridge;

  class CharacterAIExtractor extends BaseExtractor {
    constructor() {
      super('character_ai');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'textarea#user-input',
          'textarea[placeholder]',
          'textarea'
        ],
        sendSelectors: [
          'button[type="submit"]',
          'button[aria-label="Send"]',
          'button[aria-label="Send Message"]'
        ],
        inputType: 'textarea',
        messageContainer: [
          '[role="main"]',
          'main',
          '[class*="swiper"]',
          'body'
        ],
        userMessageSelectors: [
          '[data-testid="completed-message"]'
        ],
        newConversationPattern: /^\/chat\/new/,
        periodicScan: false
      };
    }

    extractUserTexts(node) {
      const texts = [];

      const messageEls = this._getMessageElements(node);
      for (const el of messageEls) {
        if (this._isUserMessage(el)) {
          const text = this._extractText(el);
          if (text && text.length >= 10) texts.push(text);
        }
      }

      return texts;
    }

    _getMessageElements(node) {
      if (node.matches?.('[data-testid="completed-message"]')) return [node];
      const els = node.querySelectorAll?.('[data-testid="completed-message"]') || [];
      return Array.from(els);
    }

    _isUserMessage(el) {
      // Character.AI: user messages have a parent with flex-row-reverse
      let parent = el.parentElement;
      for (let i = 0; i < 5 && parent; i++) {
        const cn = typeof parent.className === 'string' ? parent.className : '';
        if (cn.includes('flex-row-reverse')) return true;
        if (cn.includes('items-start') && cn.includes('flex') && cn.includes('gap-2')) return false;
        parent = parent.parentElement;
      }
      return false;
    }

    _extractText(el) {
      const proseP = el.querySelector('.prose p');
      if (proseP) return proseP.textContent.trim();
      const prose = el.querySelector('.prose');
      if (prose) return prose.textContent.trim();
      return (el.textContent || '').trim();
    }
  }

  const extractor = new CharacterAIExtractor();
  extractor.init();
})();
