/**
 * ContextBridge - Suggestion Detector
 * Regex-based pattern matching on user messages to suggest context entries.
 * No AI — pure pattern matching.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  const PATTERNS = {
    decisions: [
      { regex: /(?:I(?:'ve)?|we(?:'ve)?)\s+decided\s+to\s+(.{10,120})/i, group: 1 },
      { regex: /let'?s\s+go\s+with\s+(.{5,120})/i, group: 1 },
      { regex: /the\s+plan\s+is\s+(?:to\s+)?(.{10,120})/i, group: 1 },
      { regex: /going\s+with\s+(.{5,120})/i, group: 1 },
      { regex: /(?:I(?:'ve)?|we(?:'ve)?)\s+chosen\s+(.{5,120})/i, group: 1 },
      { regex: /final\s+decision:\s*(.{5,120})/i, group: 1 }
    ],
    preferences: [
      { regex: /I\s+prefer\s+(.{5,120})/i, group: 1 },
      { regex: /always\s+use\s+(.{5,120})/i, group: 1 },
      { regex: /never\s+use\s+(.{5,120})/i, group: 1 },
      { regex: /from\s+now\s+on,?\s+(.{5,120})/i, group: 1 },
      { regex: /I\s+like\s+(?:to\s+)?(?:use\s+)?(.{5,120})/i, group: 1 },
      { regex: /my\s+preference\s+is\s+(.{5,120})/i, group: 1 }
    ],
    projects: [
      { regex: /(?:I'm|I\s+am|we(?:'re)?)\s+working\s+on\s+(.{5,120})/i, group: 1 },
      { regex: /(?:I'm|I\s+am|we(?:'re)?)\s+building\s+(.{5,120})/i, group: 1 },
      { regex: /the\s+project\s+is\s+called\s+(.{3,80})/i, group: 1 },
      { regex: /(?:I'm|I\s+am|we(?:'re)?)\s+developing\s+(.{5,120})/i, group: 1 },
      { regex: /(?:our|my)\s+(?:project|app|product)\s+(?:is\s+)?(.{5,120})/i, group: 1 }
    ],
    rules: [
      { regex: /always\s+(.{5,120})/i, group: 0 },
      { regex: /never\s+(.{5,120})/i, group: 0 },
      { regex: /remember\s+to\s+(.{5,120})/i, group: 1 },
      { regex: /make\s+sure\s+(?:to\s+)?(.{5,120})/i, group: 1 },
      { regex: /don'?t\s+ever\s+(.{5,120})/i, group: 0 },
      { regex: /(?:you\s+)?must\s+always\s+(.{5,120})/i, group: 0 }
    ],
    people: [
      { regex: /my\s+colleague\s+(\w[\w\s]{1,40})/i, group: 0 },
      { regex: /meeting\s+with\s+(\w[\w\s]{1,40})/i, group: 0 },
      { regex: /my\s+(?:client|boss|manager|partner)\s+(?:is\s+)?(\w[\w\s]{1,40})/i, group: 0 },
      { regex: /(?:working|collaborating)\s+with\s+(\w[\w\s]{1,40})/i, group: 0 }
    ],
    tasks: [
      { regex: /(?:the\s+)?next\s+step\s+is\s+(?:to\s+)?(.{5,120})/i, group: 1 },
      { regex: /(?:I|we)\s+need\s+to\s+(.{5,120})/i, group: 1 },
      { regex: /TODO:\s*(.{5,120})/i, group: 1 },
      { regex: /(?:I|we)\s+(?:still\s+)?have\s+to\s+(.{5,120})/i, group: 1 },
      { regex: /deadline\s+(?:is\s+)?(.{5,120})/i, group: 0 }
    ]
  };

  const SuggestionDetector = {
    /**
     * Scan a user message for context suggestions.
     * @param {string} text - The user's message text
     * @returns {Array<{text: string, category: string}>} detected suggestions
     */
    scan(text) {
      if (!text || text.length < 10) return [];

      const results = [];
      const seen = new Set();

      for (const [category, patterns] of Object.entries(PATTERNS)) {
        for (const pattern of patterns) {
          const match = text.match(pattern.regex);
          if (match) {
            let extracted = (pattern.group === 0 ? match[0] : match[pattern.group]) || '';
            extracted = this._clean(extracted);

            if (extracted.length >= 5 && !seen.has(extracted.toLowerCase())) {
              seen.add(extracted.toLowerCase());
              results.push({ text: extracted, category });
            }
          }
        }
      }

      return results;
    },

    _clean(text) {
      // Trim trailing punctuation and whitespace
      return text
        .replace(/[.!?,;:]+$/, '')
        .replace(/\s+/g, ' ')
        .trim();
    }
  };

  window.ContextBridge.SuggestionDetector = SuggestionDetector;
})();
