/**
 * ContextBridge - Base Extractor
 * Abstract base class for platform extractors.
 * Handles suggestion scanning + injection lifecycle.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  const {
    SelectorEngine, ObserverManager, ContentBridge,
    ConversationTracker, SuggestionDetector, Utils
  } = window.ContextBridge;

  class BaseExtractor {
    constructor(platformKey) {
      this.platformKey = platformKey;
      this.enabled = false;
      this.autoSuggest = true;
      this.observerManager = null;
      this.scanInterval = null;
      this.containerCheckInterval = null;
      this.currentContainer = null;
      this.tracker = null;
      this.scannedMessages = new Set();
      this.scannedMaxSize = 2000;
    }

    async init() {
      console.log(`[ContextBridge] Initializing ${this.platformKey} extractor`);

      const status = await ContentBridge.checkEnabled(this.platformKey);
      if (!status.enabled) {
        console.log(`[ContextBridge] ${this.platformKey}: disabled`);
        return;
      }

      this.enabled = true;
      this.autoSuggest = status.autoSuggest !== false;

      // Get platform config
      const config = this.getPlatformConfig();

      // Initialize conversation tracker (handles injection)
      this.tracker = new ConversationTracker({
        platform: this.platformKey,
        inputSelectors: config.inputSelectors,
        sendSelectors: config.sendSelectors,
        inputType: config.inputType,
        newConversationPattern: config.newConversationPattern
      });
      await this.tracker.init();

      // Set up DOM observation for suggestion scanning
      if (this.autoSuggest) {
        await this._setupObserver();
        this.containerCheckInterval = setInterval(() => this._checkContainer(), 5000);
      }
    }

    async _setupObserver() {
      const config = this.getPlatformConfig();
      const container = config.useBodyAsContainer
        ? document.body
        : await SelectorEngine.waitFor(config.messageContainer, 15000, document);

      if (container) {
        console.log(`[ContextBridge] ${this.platformKey}: observer attached`);
        this.observerManager = new ObserverManager(this);
        this.observerManager.observe(container);
        this.currentContainer = container;

        // Periodic scan for platforms that modify DOM in-place
        if (config.periodicScan) {
          this.scanInterval = setInterval(() => this._periodicScan(), 3000);
        }
      } else {
        console.warn(`[ContextBridge] ${this.platformKey}: container not found`);
      }
    }

    _checkContainer() {
      if (!this.enabled || !this.autoSuggest) return;
      if (this.currentContainer && !document.contains(this.currentContainer)) {
        console.log(`[ContextBridge] ${this.platformKey}: container detached, re-attaching`);
        if (this.observerManager) this.observerManager.disconnect();
        this._setupObserver();
      }
    }

    /**
     * Called by ObserverManager when new nodes appear.
     * Scans user messages for suggestion patterns.
     */
    onNewNode(node) {
      if (!this.autoSuggest) return;

      const userTexts = this.extractUserTexts(node);
      for (const text of userTexts) {
        this._scanForSuggestions(text);
      }
    }

    /**
     * Periodic scan for platforms where DOM changes don't trigger mutations well.
     */
    _periodicScan() {
      if (!this.enabled || !this.autoSuggest) return;

      const config = this.getPlatformConfig();
      if (!config.userMessageSelectors) return;

      const elements = SelectorEngine.queryAll(document, config.userMessageSelectors);
      for (const el of elements) {
        const text = (el.textContent || '').trim();
        if (text.length >= 10) {
          this._scanForSuggestions(text);
        }
      }
    }

    _scanForSuggestions(text) {
      if (!text || text.length < 10) return;

      // Dedup by hash
      const hash = Utils.hash(text.substring(0, 100));
      if (this.scannedMessages.has(hash)) return;
      this.scannedMessages.add(hash);

      // Prune dedup set
      if (this.scannedMessages.size > this.scannedMaxSize) {
        const entries = Array.from(this.scannedMessages);
        this.scannedMessages = new Set(entries.slice(entries.length / 2));
      }

      // Detect suggestions
      const suggestions = SuggestionDetector.scan(text);
      for (const suggestion of suggestions) {
        ContentBridge.suggestEntry(suggestion.text, suggestion.category, this.platformKey);
        console.log(`[ContextBridge] ${this.platformKey}: suggested "${suggestion.text}" (${suggestion.category})`);
      }
    }

    destroy() {
      this.enabled = false;
      if (this.observerManager) this.observerManager.disconnect();
      if (this.scanInterval) clearInterval(this.scanInterval);
      if (this.containerCheckInterval) clearInterval(this.containerCheckInterval);
      if (this.tracker) this.tracker.destroy();
    }

    // --- Abstract methods (must be implemented by platform extractors) ---

    /**
     * Return platform-specific configuration.
     * @returns {Object} config with inputSelectors, sendSelectors, messageContainer, etc.
     */
    getPlatformConfig() {
      throw new Error('getPlatformConfig() must be implemented');
    }

    /**
     * Extract user message texts from a DOM node.
     * Used for suggestion scanning — only user messages, not AI responses.
     * @param {Node} node
     * @returns {string[]} array of user message texts
     */
    extractUserTexts(node) {
      throw new Error('extractUserTexts() must be implemented');
    }
  }

  window.ContextBridge.BaseExtractor = BaseExtractor;
})();
