/**
 * ContextBridge - Service Worker
 * Message routing between popup/content scripts and context store.
 * Zero network calls — everything local.
 */

importScripts('context-store.js');

const store = new ContextStore();

// Category ordering for context block formatting
const CATEGORY_ORDER = ['rules', 'preferences', 'projects', 'tasks', 'decisions', 'people'];
const CATEGORY_NAMES = {
  rules: 'Rules',
  preferences: 'Preferences',
  projects: 'Projects',
  tasks: 'Tasks',
  decisions: 'Decisions',
  people: 'People'
};

/**
 * Format context entries into injection block
 */
function formatContextBlock(entries, maxLength = 2000) {
  if (!entries || entries.length === 0) return '';

  const grouped = {};
  for (const entry of entries) {
    if (!grouped[entry.category]) grouped[entry.category] = [];
    grouped[entry.category].push(entry);
  }

  let block = '[CONTEXT START]\n';
  const suffix = '\n[CONTEXT END]';
  let currentLength = block.length + suffix.length;

  for (const category of CATEGORY_ORDER) {
    if (!grouped[category] || grouped[category].length === 0) continue;

    const header = `\n## ${CATEGORY_NAMES[category]}\n`;
    if (currentLength + header.length > maxLength) break;

    block += header;
    currentLength += header.length;

    for (const entry of grouped[category]) {
      const line = `- ${entry.text}\n`;
      if (currentLength + line.length > maxLength) break;
      block += line;
      currentLength += line.length;
    }
  }

  block += suffix;
  return block;
}

// Message handler
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender).then(sendResponse);
  return true; // async response
});

async function handleMessage(message, sender) {
  const { type, payload } = message;

  try {
    switch (type) {
      // --- Context ---
      case 'GET_CONTEXT': {
        const entries = await store.getEntries();
        const settings = await store.getSettings();
        const block = formatContextBlock(entries, settings.maxContextLength);
        return { success: true, block, entries, settings };
      }

      case 'FORMAT_CONTEXT': {
        const entries = await store.getEntries();
        const settings = await store.getSettings();
        const block = formatContextBlock(entries, settings.maxContextLength);
        return { success: true, block };
      }

      case 'GET_ALL_ENTRIES': {
        const entries = await store.getEntries();
        return { success: true, entries };
      }

      case 'ADD_ENTRY': {
        const entry = await store.addEntry(payload);
        return { success: true, entry };
      }

      case 'UPDATE_ENTRY': {
        const entry = await store.updateEntry(payload.id, payload.updates);
        return { success: !!entry, entry };
      }

      case 'DELETE_ENTRY': {
        const deleted = await store.deleteEntry(payload.id);
        return { success: deleted };
      }

      // --- Suggestions ---
      case 'SUGGEST_ENTRY': {
        const suggestion = await store.addSuggestion(payload);
        if (suggestion) {
          // Update badge to show pending suggestions count
          const suggestions = await store.getSuggestions();
          const count = suggestions.length;
          if (count > 0) {
            chrome.action.setBadgeText({ text: String(count) });
            chrome.action.setBadgeBackgroundColor({ color: '#6366f1' });
          }
        }
        return { success: !!suggestion, suggestion };
      }

      case 'GET_SUGGESTIONS': {
        const suggestions = await store.getSuggestions();
        return { success: true, suggestions };
      }

      case 'APPROVE_SUGGESTION': {
        const entry = await store.approveSuggestion(payload.id);
        // Update badge
        const remaining = await store.getSuggestions();
        if (remaining.length === 0) {
          chrome.action.setBadgeText({ text: '' });
        } else {
          chrome.action.setBadgeText({ text: String(remaining.length) });
        }
        return { success: !!entry, entry };
      }

      case 'DISMISS_SUGGESTION': {
        const dismissed = await store.dismissSuggestion(payload.id);
        const remaining = await store.getSuggestions();
        if (remaining.length === 0) {
          chrome.action.setBadgeText({ text: '' });
        } else {
          chrome.action.setBadgeText({ text: String(remaining.length) });
        }
        return { success: dismissed };
      }

      // --- Settings ---
      case 'CHECK_ENABLED': {
        const settings = await store.getSettings();
        const platform = payload.platform;
        const enabled = settings.platformsEnabled?.[platform] !== false;
        return { success: true, enabled, autoInject: settings.autoInject, autoSuggest: settings.autoSuggest };
      }

      case 'GET_SETTINGS': {
        const settings = await store.getSettings();
        return { success: true, settings };
      }

      case 'SAVE_SETTINGS': {
        const settings = await store.saveSettings(payload);
        return { success: true, settings };
      }

      // --- Stats ---
      case 'GET_STATS': {
        const stats = await store.getStats();
        const entries = await store.getEntries();
        const suggestions = await store.getSuggestions();
        return { success: true, stats, entryCount: entries.length, suggestionCount: suggestions.length };
      }

      case 'RECORD_INJECTION': {
        const stats = await store.recordInjection(payload.platform);
        return { success: true, stats };
      }

      // --- Export / Import ---
      case 'EXPORT_DATA': {
        const data = await store.exportAll();
        return { success: true, data };
      }

      case 'IMPORT_DATA': {
        await store.importAll(payload);
        return { success: true };
      }

      case 'CLEAR_ALL': {
        await store.clearAll();
        chrome.action.setBadgeText({ text: '' });
        return { success: true };
      }

      default:
        return { success: false, error: `Unknown message type: ${type}` };
    }
  } catch (err) {
    console.error('[ContextBridge] Service worker error:', err);
    return { success: false, error: err.message };
  }
}

// Initialize badge on install
chrome.runtime.onInstalled.addListener(async () => {
  console.log('[ContextBridge] Extension installed');
  chrome.action.setBadgeBackgroundColor({ color: '#6366f1' });
});
