/**
 * ContextBridge - DeepSeek Extractor
 * chat.deepseek.com
 * .ds-message contains BOTH user text and nested AI response (.ds-markdown).
 * Must clone and strip markdown to get pure user text.
 */

(function () {
  const { BaseExtractor, SelectorEngine } = window.ContextBridge;

  class DeepSeekExtractor extends BaseExtractor {
    constructor() {
      super('deepseek');
    }

    getPlatformConfig() {
      return {
        inputSelectors: [
          'textarea#chat-input',
          'textarea[placeholder]',
          'textarea'
        ],
        sendSelectors: [
          'div.ds-button',
          'button[aria-label="Send"]',
          'button[type="submit"]'
        ],
        inputType: 'textarea',
        messageContainer: ['body'],
        useBodyAsContainer: true,
        userMessageSelectors: [
          '.ds-message'
        ],
        newConversationPattern: /^\/($|chat\/?$|new)/,
        periodicScan: true
      };
    }

    extractUserTexts(node) {
      const texts = [];

      const messageEls = this._getMessageElements(node);
      for (const el of messageEls) {
        const text = this._extractUserText(el);
        if (text && text.length >= 10) texts.push(text);
      }

      return texts;
    }

    _getMessageElements(node) {
      if (node.matches?.('.ds-message')) return [node];
      const els = node.querySelectorAll?.('.ds-message') || [];
      return Array.from(els);
    }

    _extractUserText(el) {
      // Clone and strip AI markdown to get pure user text
      const hasMarkdown = el.querySelector('.ds-markdown');
      if (!hasMarkdown) {
        // No AI response nested — this is a user-only message
        return (el.textContent || '').trim();
      }

      // Clone, remove markdown, get remaining text
      const clone = el.cloneNode(true);
      const markdowns = clone.querySelectorAll('.ds-markdown');
      for (const md of markdowns) md.remove();
      const text = (clone.textContent || '').trim();

      // If no meaningful text remains, it's an AI-only message
      return text.length >= 5 ? text : null;
    }
  }

  const extractor = new DeepSeekExtractor();
  extractor.init();
})();
