/**
 * ContextBridge - Selector Engine
 * Resilient CSS selector resolution with fallback chains.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  class SelectorEngine {
    static queryFirst(root, selectorChain) {
      const selectors = Array.isArray(selectorChain) ? selectorChain : [selectorChain];
      for (const selector of selectors) {
        try {
          const el = root.querySelector(selector);
          if (el) return el;
        } catch (e) {}
      }
      return null;
    }

    static queryAll(root, selectorChain) {
      const selectors = Array.isArray(selectorChain) ? selectorChain : [selectorChain];
      for (const selector of selectors) {
        try {
          const els = root.querySelectorAll(selector);
          if (els.length > 0) return Array.from(els);
        } catch (e) {}
      }
      return [];
    }

    static getText(root, selectorChain) {
      const el = this.queryFirst(root, selectorChain);
      return el ? el.textContent.trim() : '';
    }

    static waitFor(selectorChain, timeoutMs = 10000, root = document) {
      return new Promise((resolve) => {
        const existing = this.queryFirst(root, selectorChain);
        if (existing) { resolve(existing); return; }

        const observer = new MutationObserver(() => {
          const el = this.queryFirst(root, selectorChain);
          if (el) {
            observer.disconnect();
            clearTimeout(timer);
            resolve(el);
          }
        });

        const timer = setTimeout(() => {
          observer.disconnect();
          resolve(null);
        }, timeoutMs);

        observer.observe(root, { childList: true, subtree: true });
      });
    }
  }

  window.ContextBridge.SelectorEngine = SelectorEngine;
})();
