/**
 * ContextBridge - Observer Manager
 * Throttled MutationObserver — triggers suggestion scanning on new user messages.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  class ObserverManager {
    constructor(extractor) {
      this.extractor = extractor;
      this.observer = null;
      this.pendingMutations = [];
      this.flushScheduled = false;
      this.throttleMs = window.ContextBridge.CB_TIMING
        ? window.ContextBridge.CB_TIMING.OBSERVER_THROTTLE_MS : 500;
    }

    observe(targetNode) {
      this.disconnect();
      this.observer = new MutationObserver((mutations) => {
        this.pendingMutations.push(...mutations);
        if (!this.flushScheduled) {
          this.flushScheduled = true;
          setTimeout(() => this.processMutations(), this.throttleMs);
        }
      });
      this.observer.observe(targetNode, { childList: true, subtree: true });
    }

    processMutations() {
      const mutations = this.pendingMutations.splice(0);
      this.flushScheduled = false;

      for (const mutation of mutations) {
        for (const addedNode of mutation.addedNodes) {
          if (addedNode.nodeType !== Node.ELEMENT_NODE) continue;
          try {
            this.extractor.onNewNode(addedNode);
          } catch (err) {
            console.warn('[ContextBridge] Observer error:', err.message);
          }
        }
      }
    }

    disconnect() {
      if (this.observer) {
        this.observer.disconnect();
        this.observer = null;
      }
      this.pendingMutations = [];
      this.flushScheduled = false;
    }
  }

  window.ContextBridge.ObserverManager = ObserverManager;
})();
