/**
 * ContextBridge - Injection Engine
 * Writes text into platform input fields.
 * Handles contenteditable divs (ProseMirror), textareas, and custom elements.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  const InjectionEngine = {

    /**
     * Inject text into an input element, prepending to existing content.
     * @param {HTMLElement} inputEl - The input element
     * @param {string} contextBlock - The context block to prepend
     * @param {string} inputType - 'contenteditable' | 'textarea' | 'custom'
     * @returns {boolean} success
     */
    inject(inputEl, contextBlock, inputType = 'auto') {
      if (!inputEl || !contextBlock) return false;

      const type = inputType === 'auto' ? this._detectType(inputEl) : inputType;

      try {
        switch (type) {
          case 'contenteditable':
            return this._injectContentEditable(inputEl, contextBlock);
          case 'textarea':
            return this._injectTextarea(inputEl, contextBlock);
          case 'custom':
            return this._injectCustom(inputEl, contextBlock);
          default:
            console.warn('[ContextBridge] Unknown input type:', type);
            return false;
        }
      } catch (err) {
        console.error('[ContextBridge] Injection failed:', err);
        return false;
      }
    },

    /**
     * Read current text from input element
     */
    readInput(inputEl) {
      if (!inputEl) return '';
      const type = this._detectType(inputEl);

      if (type === 'textarea') {
        return inputEl.value || '';
      }
      // contenteditable or custom
      return inputEl.innerText || inputEl.textContent || '';
    },

    /**
     * Write full text into input element (replacing existing content)
     */
    writeInput(inputEl, text) {
      if (!inputEl) return false;
      const type = this._detectType(inputEl);

      try {
        if (type === 'textarea') {
          return this._setTextareaValue(inputEl, text);
        }
        return this._setContentEditableValue(inputEl, text);
      } catch (err) {
        console.error('[ContextBridge] Write failed:', err);
        return false;
      }
    },

    _detectType(el) {
      if (!el) return 'unknown';
      const tag = el.tagName?.toLowerCase();
      if (tag === 'textarea') return 'textarea';
      if (tag === 'input') return 'textarea';
      if (el.getAttribute?.('contenteditable') === 'true') return 'contenteditable';
      // Custom web components (e.g., Gemini's rich-textarea)
      if (tag && tag.includes('-')) return 'custom';
      return 'contenteditable';
    },

    /**
     * Contenteditable injection (ChatGPT, Claude — ProseMirror editors)
     * execCommand is deprecated but functional in Chromium and crucially
     * triggers React/ProseMirror input event handlers.
     */
    _injectContentEditable(el, contextBlock) {
      const existing = el.innerText || '';
      const fullText = contextBlock + '\n\n' + existing;

      el.focus();
      document.execCommand('selectAll', false, null);
      document.execCommand('insertText', false, fullText);

      // Verify
      const result = el.innerText || '';
      console.log(`[ContextBridge] CE injection: ${result.length} chars written`);
      return result.length > 0;
    },

    /**
     * Textarea injection (Pi, Poe, Character.AI, HuggingChat, etc.)
     * Must bypass React's synthetic value property.
     */
    _injectTextarea(el, contextBlock) {
      const existing = el.value || '';
      const fullText = contextBlock + '\n\n' + existing;
      return this._setTextareaValue(el, fullText);
    },

    _setTextareaValue(el, text) {
      const setter = Object.getOwnPropertyDescriptor(
        HTMLTextAreaElement.prototype, 'value'
      )?.set;

      if (setter) {
        setter.call(el, text);
      } else {
        el.value = text;
      }

      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));

      console.log(`[ContextBridge] Textarea injection: ${text.length} chars written`);
      return true;
    },

    _setContentEditableValue(el, text) {
      el.focus();
      document.execCommand('selectAll', false, null);
      document.execCommand('insertText', false, text);
      return true;
    },

    /**
     * Custom element injection (Gemini rich-textarea)
     * Drill into shadow DOM or inner contenteditable/textarea.
     */
    _injectCustom(el, contextBlock) {
      // Try shadow DOM
      const shadow = el.shadowRoot;
      if (shadow) {
        const inner = shadow.querySelector('textarea') ||
                      shadow.querySelector('[contenteditable="true"]');
        if (inner) {
          const type = this._detectType(inner);
          return type === 'textarea'
            ? this._injectTextarea(inner, contextBlock)
            : this._injectContentEditable(inner, contextBlock);
        }
      }

      // Try inner elements
      const innerTextarea = el.querySelector('textarea');
      if (innerTextarea) {
        return this._injectTextarea(innerTextarea, contextBlock);
      }

      const innerCE = el.querySelector('[contenteditable="true"]');
      if (innerCE) {
        return this._injectContentEditable(innerCE, contextBlock);
      }

      // Fallback: treat as contenteditable
      if (el.getAttribute?.('contenteditable') === 'true') {
        return this._injectContentEditable(el, contextBlock);
      }

      console.warn('[ContextBridge] Custom element: no injectable child found');
      return false;
    }
  };

  window.ContextBridge.InjectionEngine = InjectionEngine;
})();
