/**
 * ContextBridge - Conversation Tracker
 * Detects new conversations and intercepts first send to inject context.
 * Belt-and-suspenders: popstate + history monkey-patch + URL polling.
 */

(function () {
  window.ContextBridge = window.ContextBridge || {};

  const { ContentBridge, InjectionEngine, SelectorEngine } = window.ContextBridge;

  class ConversationTracker {
    constructor(platformConfig) {
      this.platform = platformConfig.platform;
      this.inputSelectors = platformConfig.inputSelectors;
      this.sendSelectors = platformConfig.sendSelectors;
      this.inputType = platformConfig.inputType || 'auto';
      this.newConversationPattern = platformConfig.newConversationPattern || null;

      this.contextPending = false;
      this.currentUrl = window.location.href;
      this.urlPollInterval = null;
      this.sendListenerAttached = false;
      this.enabled = false;
      this.autoInject = true;
    }

    async init() {
      const status = await ContentBridge.checkEnabled(this.platform);
      if (!status.enabled) {
        console.log(`[ContextBridge] ${this.platform}: disabled`);
        return;
      }

      this.enabled = true;
      this.autoInject = status.autoInject !== false;

      if (!this.autoInject) {
        console.log(`[ContextBridge] ${this.platform}: auto-inject disabled`);
        return;
      }

      // Determine if this is a new conversation on page load
      if (this._isNewConversation()) {
        this.contextPending = true;
        console.log(`[ContextBridge] ${this.platform}: new conversation detected on load`);
      }

      this._setupUrlTracking();
      this._attachSendInterceptor();

      console.log(`[ContextBridge] ${this.platform}: conversation tracker initialized`);
    }

    // --- URL change detection (belt-and-suspenders) ---

    _setupUrlTracking() {
      // 1. popstate
      window.addEventListener('popstate', () => this._onUrlChange());

      // 2. Monkey-patch pushState / replaceState
      const self = this;
      const origPush = history.pushState;
      const origReplace = history.replaceState;

      history.pushState = function (...args) {
        origPush.apply(this, args);
        self._onUrlChange();
      };

      history.replaceState = function (...args) {
        origReplace.apply(this, args);
        self._onUrlChange();
      };

      // 3. Polling fallback (1s)
      this.urlPollInterval = setInterval(() => {
        if (window.location.href !== this.currentUrl) {
          this._onUrlChange();
        }
      }, 1000);
    }

    _onUrlChange() {
      const newUrl = window.location.href;
      if (newUrl === this.currentUrl) return;

      console.log(`[ContextBridge] ${this.platform}: URL changed to ${newUrl}`);
      this.currentUrl = newUrl;

      if (this._isNewConversation()) {
        this.contextPending = true;
        this.sendListenerAttached = false;
        console.log(`[ContextBridge] ${this.platform}: new conversation — context pending`);
        this._attachSendInterceptor();
      }
    }

    _isNewConversation() {
      const path = window.location.pathname;

      // If a platform-specific pattern is provided, use it
      if (this.newConversationPattern) {
        return this.newConversationPattern.test(path);
      }

      // Default: root paths or paths without conversation IDs are "new"
      // Most platforms: / or /chat or /new
      if (path === '/' || path.endsWith('/new') || path.endsWith('/chat')) return true;

      // Heuristic: paths with UUIDs or long IDs are existing conversations
      if (/[a-f0-9-]{20,}/.test(path)) return false;

      return true;
    }

    // --- Send button interception ---

    _attachSendInterceptor() {
      if (this.sendListenerAttached) return;

      // Wait for send button to appear, then attach
      const tryAttach = () => {
        const sendBtn = SelectorEngine.queryFirst(document, this.sendSelectors);
        if (sendBtn) {
          this._attachToButton(sendBtn);
          return true;
        }
        return false;
      };

      if (!tryAttach()) {
        // Retry with observer
        let attempts = 0;
        const retryInterval = setInterval(() => {
          attempts++;
          if (tryAttach() || attempts > 30) {
            clearInterval(retryInterval);
          }
        }, 500);
      }
    }

    _attachToButton(sendBtn) {
      if (this.sendListenerAttached) return;
      this.sendListenerAttached = true;

      // Capture phase — fires before platform's handler
      sendBtn.addEventListener('click', (e) => this._onSendClick(e), true);

      // Also watch for Enter key on the input
      const inputEl = SelectorEngine.queryFirst(document, this.inputSelectors);
      if (inputEl) {
        inputEl.addEventListener('keydown', (e) => {
          if (e.key === 'Enter' && !e.shiftKey) {
            this._onSendKey(e);
          }
        }, true);
      }

      console.log(`[ContextBridge] ${this.platform}: send interceptor attached`);
    }

    async _onSendClick(e) {
      if (!this.contextPending) return;
      await this._interceptSend(e);
    }

    async _onSendKey(e) {
      if (!this.contextPending) return;
      await this._interceptSend(e);
    }

    async _interceptSend(e) {
      if (!this.contextPending) return;

      const inputEl = SelectorEngine.queryFirst(document, this.inputSelectors);
      if (!inputEl) {
        console.warn(`[ContextBridge] ${this.platform}: input element not found for injection`);
        return;
      }

      // Get current message text
      const currentText = InjectionEngine.readInput(inputEl);
      if (!currentText || currentText.trim().length === 0) return;

      // Get context block from service worker
      const response = await ContentBridge.getContext();
      if (!response.success || !response.block || response.block.trim().length === 0) {
        // No context to inject
        this.contextPending = false;
        return;
      }

      // Prevent the original send
      e.preventDefault();
      e.stopPropagation();

      // Prepend context block to user's message
      const fullText = response.block + '\n\n' + currentText.trim();

      // Write the combined text back to the input
      InjectionEngine.writeInput(inputEl, fullText);

      // Mark injection done
      this.contextPending = false;
      ContentBridge.recordInjection(this.platform);

      console.log(`[ContextBridge] ${this.platform}: context injected (${response.block.length} chars)`);

      // Re-trigger send after a brief delay for the framework to process the input
      setTimeout(() => {
        const sendBtn = SelectorEngine.queryFirst(document, this.sendSelectors);
        if (sendBtn) {
          sendBtn.click();
        }
      }, 100);
    }

    destroy() {
      this.enabled = false;
      if (this.urlPollInterval) {
        clearInterval(this.urlPollInterval);
        this.urlPollInterval = null;
      }
    }
  }

  window.ContextBridge.ConversationTracker = ConversationTracker;
})();
