/**
 * ContextBridge - Context Store
 * CRUD operations for entries, suggestions, settings, and stats.
 * All data in chrome.storage.local — zero network calls.
 */

const CB_STORAGE_DEFAULTS = {
  cb_entries: [],
  cb_suggestions: [],
  cb_settings: {
    autoInject: true,
    autoSuggest: true,
    maxContextLength: 2000,
    platformsEnabled: {
      chatgpt: true,
      claude: true,
      gemini: true,
      copilot: true,
      character_ai: true,
      deepseek: true,
      grok: true,
      pi: true,
      poe: true,
      perplexity: true,
      huggingchat: true,
      lechat: true
    }
  },
  cb_stats: {
    totalInjections: 0,
    lastInjection: null
  }
};

class ContextStore {

  // --- Entries ---

  async getEntries() {
    const result = await chrome.storage.local.get('cb_entries');
    return result.cb_entries || [];
  }

  async addEntry(entry) {
    const entries = await this.getEntries();
    const newEntry = {
      id: this._generateId(),
      text: entry.text,
      category: entry.category,
      sourcePlatform: entry.sourcePlatform || 'manual',
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    entries.push(newEntry);
    await chrome.storage.local.set({ cb_entries: entries });
    return newEntry;
  }

  async updateEntry(id, updates) {
    const entries = await this.getEntries();
    const idx = entries.findIndex(e => e.id === id);
    if (idx === -1) return null;
    entries[idx] = { ...entries[idx], ...updates, updatedAt: new Date().toISOString() };
    await chrome.storage.local.set({ cb_entries: entries });
    return entries[idx];
  }

  async deleteEntry(id) {
    const entries = await this.getEntries();
    const filtered = entries.filter(e => e.id !== id);
    await chrome.storage.local.set({ cb_entries: filtered });
    return filtered.length < entries.length;
  }

  // --- Suggestions ---

  async getSuggestions() {
    const result = await chrome.storage.local.get('cb_suggestions');
    return result.cb_suggestions || [];
  }

  async addSuggestion(suggestion) {
    const suggestions = await this.getSuggestions();

    // Cap at 50
    if (suggestions.length >= 50) {
      suggestions.shift();
    }

    // Deduplicate by text
    const exists = suggestions.some(s => s.text === suggestion.text);
    if (exists) return null;

    const newSuggestion = {
      id: this._generateId(),
      text: suggestion.text,
      category: suggestion.category,
      sourcePlatform: suggestion.sourcePlatform || 'unknown',
      suggestedAt: new Date().toISOString()
    };
    suggestions.push(newSuggestion);
    await chrome.storage.local.set({ cb_suggestions: suggestions });
    return newSuggestion;
  }

  async approveSuggestion(id) {
    const suggestions = await this.getSuggestions();
    const idx = suggestions.findIndex(s => s.id === id);
    if (idx === -1) return null;

    const suggestion = suggestions[idx];
    suggestions.splice(idx, 1);
    await chrome.storage.local.set({ cb_suggestions: suggestions });

    // Add as entry
    return await this.addEntry({
      text: suggestion.text,
      category: suggestion.category,
      sourcePlatform: suggestion.sourcePlatform
    });
  }

  async dismissSuggestion(id) {
    const suggestions = await this.getSuggestions();
    const filtered = suggestions.filter(s => s.id !== id);
    await chrome.storage.local.set({ cb_suggestions: filtered });
    return filtered.length < suggestions.length;
  }

  // --- Settings ---

  async getSettings() {
    const result = await chrome.storage.local.get('cb_settings');
    return { ...CB_STORAGE_DEFAULTS.cb_settings, ...(result.cb_settings || {}) };
  }

  async saveSettings(settings) {
    const current = await this.getSettings();
    const merged = { ...current, ...settings };
    await chrome.storage.local.set({ cb_settings: merged });
    return merged;
  }

  // --- Stats ---

  async getStats() {
    const result = await chrome.storage.local.get('cb_stats');
    return { ...CB_STORAGE_DEFAULTS.cb_stats, ...(result.cb_stats || {}) };
  }

  async recordInjection(platform) {
    const stats = await this.getStats();
    stats.totalInjections++;
    stats.lastInjection = new Date().toISOString();
    stats.lastPlatform = platform;
    await chrome.storage.local.set({ cb_stats: stats });
    return stats;
  }

  // --- Export / Import ---

  async exportAll() {
    const result = await chrome.storage.local.get(['cb_entries', 'cb_suggestions', 'cb_settings', 'cb_stats']);
    return {
      cb_entries: result.cb_entries || [],
      cb_suggestions: result.cb_suggestions || [],
      cb_settings: result.cb_settings || CB_STORAGE_DEFAULTS.cb_settings,
      cb_stats: result.cb_stats || CB_STORAGE_DEFAULTS.cb_stats,
      exportedAt: new Date().toISOString(),
      version: '1.0.0'
    };
  }

  async importAll(data) {
    if (!data || !data.cb_entries) throw new Error('Invalid import data');
    await chrome.storage.local.set({
      cb_entries: data.cb_entries || [],
      cb_suggestions: data.cb_suggestions || [],
      cb_settings: data.cb_settings || CB_STORAGE_DEFAULTS.cb_settings,
      cb_stats: data.cb_stats || CB_STORAGE_DEFAULTS.cb_stats
    });
    return true;
  }

  async clearAll() {
    await chrome.storage.local.set({
      cb_entries: [],
      cb_suggestions: [],
      cb_stats: CB_STORAGE_DEFAULTS.cb_stats
    });
    return true;
  }

  // --- Helpers ---

  _generateId() {
    return 'cb_' + Date.now().toString(36) + '_' + Math.random().toString(36).substring(2, 8);
  }
}
